/**
    \file DeadReckoning_7x7.cpp
    Implementation of DeadReckoning_7x7 distance transform class which, given
    and input binary image, calculates the corresponding distance transform.

    \author George J. Grevera, Ph.D., ggrevera@sju.edu

    Copyright (C) 2002, George J. Grevera

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
    USA or from http://www.gnu.org/licenses/gpl.txt.

    This General Public License does not permit incorporating this
    code into proprietary programs.  (So a hypothetical company such
    as GH (Generally Hectic) should NOT incorporate this code into
    their proprietary programs.)
 */
/*
     no OUT OF BOUNDS checks!
*/
#include <stdlib.h>
#include "DeadReckoning_7x7.h"

using namespace std;
//----------------------------------------------------------------------
/*
Input : (C,I) - a 2D binary scene of domain size X by Y
Output: (C,d) - a 2D grey scene of domain size X and Y representing the
                distance scene
*/
void DeadReckoning_7x7::doTransform ( const unsigned char* const I ) {
    const double dx=1.0;
    const double dy=1.0;
    borderCheck(I);
    cleanUp();
    int x, y;

    //initialize d
    double* d = (double*)malloc( ySize*xSize*sizeof(double)   );
    assert( d!=NULL );
    this->p = (P*)malloc( ySize*xSize*sizeof(P) );
    assert( this->p!=NULL );
    for (y=0; y<ySize; y++) {
        for (x=0; x<xSize; x++) {
            const int i=sub(x,y);
            d[i] = DistanceTransform::FloatInfinity;
            //pointer to immediate interior or exterior i.e.
            //p(x,y).x is the x coordinate of the closest boundary
            //point (and similarly for p(x,y).y)
            this->p[i].x = this->p[i].y = -1;
        }
    }

    //initialize immediate interior & exterior elements
    for (y=1; y<ySize-1; y++) {
        for (x=1; x<xSize-1; x++) {
            if ( I[sub(x-1,y)] != I[sub(x,y)] || I[sub(x+1,y)] != I[sub(x,y)] ||
                 I[sub(x,y-1)] != I[sub(x,y)] || I[sub(x,y+1)] != I[sub(x,y)] )
            {
                    const int i=sub(x,y);
                    d[i] = 0;
                    this->p[i].x = x;
                    this->p[i].y = y;
            }
        }
    }

    const double dxy = sqrt( 2.0);  //diagonal distance
    const double C   = sqrt( 5.0);
    const double D   = sqrt(10.0);
    const double E   = sqrt(13.0);

        #define _check(X,Y,Delta)                             \
            i1=sub((X),(Y));                                  \
            if (d[i1]+(Delta) < d[i2]) {                      \
                this->p[i2] = this->p[i1];                    \
                const double deltaX = (p[i1].x - x);          \
                const double deltaY = (p[i1].y - y);          \
                d[i2] = sqrt(deltaX*deltaX + deltaY*deltaY);  \
            }

        //perform the first pass
        for (y=3; y<ySize-3; y++) {
            for (x=3; x<xSize-3; x++) {
                int i1;
                const int i2=sub(x,y);

                _check( x-1, y-1, dxy );
                _check( x,   y-1, dy  );
                _check( x+1, y-1, dxy );
                _check( x-1, y,   dx  );

                _check( x-1, y-2, C );
                _check( x+1, y-2, C );
                _check( x-2, y-1, C );
                _check( x+2, y-1, C );

                _check( x-1, y-3, D );
                _check( x+1, y-3, D );
                _check( x-3, y-1, D );
                _check( x+3, y-1, D );

                _check( x-2, y-3, E );
                _check( x+2, y-3, E );
                _check( x-3, y-2, E );
                _check( x+3, y-2, E );
            }
        }

        //perform the final pass
        for (y=ySize-4; y>=3; y--) {
            for (x=xSize-4; x>=3; x--) {
                int i1;
                const int i2=sub(x,y);

                _check( x+1, y,   dx  );
                _check( x-1, y+1, dxy );
                _check( x,   y+1, dy  );
                _check( x+1, y+1, dxy );

                _check( x-2, y+1, C );
                _check( x+2, y+1, C );
                _check( x-1, y+2, C );
                _check( x+1, y+2, C );

                _check( x-1, y+3, D );
                _check( x+1, y+3, D );
                _check( x-3, y+1, D );
                _check( x+3, y+1, D );

                _check( x-2, y+3, E );
                _check( x+2, y+3, E );
                _check( x-3, y+2, E );
                _check( x+3, y+2, E );
            }
        }

    finish(I, d);
}
//----------------------------------------------------------------------
void DeadReckoning_7x7::borderCheck ( const unsigned char* const I ) {
    for (int x=0; x<xSize; x++) {
        assert( I[sub(x,0)]==0 );        //check the first row
        assert( I[sub(x,1)]==0 );        //check the second row
        assert( I[sub(x,2)]==0 );        //check the third row

        assert( I[sub(x,ySize-2)]==0 );  //check the next-to-next-to-last row
        assert( I[sub(x,ySize-2)]==0 );  //check the next-to-last row
        assert( I[sub(x,ySize-1)]==0 );  //check the last row
    }

    for (int y=0; y<ySize; y++) {
        assert( I[sub(0,y)]==0 );        //check the first col
        assert( I[sub(1,y)]==0 );        //check the second col
        assert( I[sub(2,y)]==0 );        //check the third col

        assert( I[sub(xSize-3,y)]==0 );  //check the next-to-next-to-last col
        assert( I[sub(xSize-2,y)]==0 );  //check the next-to-last col
        assert( I[sub(xSize-1,y)]==0 );  //check the last col
    }
}
//----------------------------------------------------------------------

